<?php

namespace Common\Service;

use Common\Exception;
use Common\Stdlib\StrUtils;
use Common\db\SequenceGeneratorInterface;

/**
 * Class IdGeneratorService can be used to generate new uris for the items.
 *
 * @package Common\Service
 */
class IdService
{
    const KEY_DOMAIN_PREFIX = 'DOMAIN_PREFIX';
    const KEY_DOMAIN = 'DOMAIN';
    const KEY_NS = 'NS';
    const KEY_OID = 'OID';

    const DOMAIN_LUOMUS = 'http://id.luomus.fi/';
    const QNAME_PREFIX_LUOMUS = 'luomus:';

    const DOMAIN_TUN = 'http://tun.fi/';
    const QNAME_PREFIX_TUN = 'tun:';

    const DOMAIN_ZMUO = 'http://id.zmuo.oulu.fi/';
    const QNAME_PREFIX_ZMUO = 'zmuo:';

    const DOMAIN_HERBO = 'http://id.herb.oulu.fi/';
    const QNAME_PREFIX_HERBO = 'herbo:';

    const DOMAIN_UTU = 'http://mus.utu.fi/';
    const QNAME_PREFIX_UTU = 'utu:';

    const DOMAIN_XSD = 'https://www.w3.org/TR/xmlschema-2/#';
    const QNAME_PREFIX_XSD = 'xsd:';

    const DOMAIN_RDF = 'http://www.w3.org/1999/02/22-rdf-syntax-ns#';
    const QNAME_PREFIX_RDF = 'rdf:';

    const DOMAIN_RDFS = 'http://www.w3.org/2000/01/rdf-schema#';
    const QNAME_PREFIX_RDFS = 'rdfs:';

    const DOMAIN_XML = 'http://www.w3.org/XML/1998/namespace#';
    const QNAME_PREFIX_XML = 'xml:';

    const DOMAIN_OWL = 'http://www.w3.org/2002/07/owl#';
    const QNAME_PREFIX_OWL = 'owl:';

    const DOMAIN_DC = 'http://purl.org/dc/terms/';
    const QNAME_PREFIX_DC = 'dc:';

    const DOMAIN_DWC = 'http://rs.tdwg.org/dwc/terms/';
    const QNAME_PREFIX_DWC = 'dwc:';

    const DOMAIN_DWCTYPE = 'http://rs.tdwg.org/dwc/dwctype/';
    const QNAME_PREFIX_DWCTYPE = 'dwctype:';

    const DOMAIN_ABCD = 'http://www.tdwg.org/schemas/abcd/2.06#';
    const QNAME_PREFIX_ABCD = 'abcd:';

    const DOMAIN_NATURFORSKAREN = 'http://naturforskaren.se/';
    const QNAME_PREFIX_NATURFORSKAREN = 'naturforskaren:';

    const DOMAIN_EOL = 'http://eol.org/';
    const QNAME_PREFIX_EOL = 'eol:';

    const DOMAIN_SKOS = 'http://www.w3.org/2004/02/skos/core#';
    const QNAME_PREFIX_SKOS = 'skos:';

    const DOMAIN_DYNTAXA = 'http://dyntaxa.se/Taxon/Info/';
    const QNAME_PREFIX_DYNTAXA = 'dyntaxa:';

    const DOMAIN_TAXONID = 'http://taxonid.org/';
    const QNAME_PREFIX_TAXONID = 'taxonid:';

    private static $conversions = [
        self::QNAME_PREFIX_TUN => self::DOMAIN_TUN,
        self::QNAME_PREFIX_LUOMUS => self::DOMAIN_LUOMUS,
        self::QNAME_PREFIX_ZMUO => self::DOMAIN_ZMUO,
        self::QNAME_PREFIX_HERBO => self::DOMAIN_HERBO,
        self::QNAME_PREFIX_UTU => self::DOMAIN_UTU,
        self::QNAME_PREFIX_XSD => self::DOMAIN_XSD,
        self::QNAME_PREFIX_RDF => self::DOMAIN_RDF,
        self::QNAME_PREFIX_RDFS => self::DOMAIN_RDFS,
        self::QNAME_PREFIX_XML => self::DOMAIN_XML,
        self::QNAME_PREFIX_OWL => self::DOMAIN_OWL,
        self::QNAME_PREFIX_DWC => self::DOMAIN_DWC,
        self::QNAME_PREFIX_DC => self::DOMAIN_DC,
        self::QNAME_PREFIX_DWCTYPE => self::DOMAIN_DWCTYPE,
        self::QNAME_PREFIX_ABCD => self::DOMAIN_ABCD,
        self::QNAME_PREFIX_NATURFORSKAREN => self::DOMAIN_NATURFORSKAREN,
        self::QNAME_PREFIX_EOL => self::DOMAIN_EOL,
        self::QNAME_PREFIX_SKOS => self::DOMAIN_SKOS,
        self::QNAME_PREFIX_DYNTAXA => self::DOMAIN_DYNTAXA,
        self::QNAME_PREFIX_TAXONID => self::DOMAIN_TAXONID,
    ];

    const DOMAIN_PREFIXES_PART = '(?P<'.self::KEY_DOMAIN_PREFIX.'>'.
    self::QNAME_PREFIX_LUOMUS . '|' .
    self::QNAME_PREFIX_TUN . '|' .
    self::QNAME_PREFIX_ZMUO . '|' .
    self::QNAME_PREFIX_UTU . '|' .
    '(?P<noNS>' .self::QNAME_PREFIX_HERBO . '))';
    const DOMAIN_PART = 'http:\/\/(?P<' . self::KEY_DOMAIN . '>(mus.utu\.fi|tun\.fi|id\.(luomus|zmuo\.oulu|(?P<noNS>herb\.oulu))\.fi))\/';

    const NS_PART = '(?P<' . self::KEY_NS . '>[a-z0-9]{1,4})';
    const NS_WITH_DOMAIN_PART = '(' . self::DOMAIN_PREFIXES_PART . ')?' . self::NS_PART;
    const OID_PART = '(?P<'. self::KEY_OID .'>([a-z]+[\-]?)?[a-z0-9#]+)';
    const QNAME_PART = self::NS_PART . '*?(?(<noNS>)|\.)' . self::OID_PART;
    const QNAME_WITH_DOMAIN_PART = self::DOMAIN_PREFIXES_PART . '?' . self::QNAME_PART;

    const NS_REGEXP = '/^' . self::NS_WITH_DOMAIN_PART . '$/i';
    const ID_RANGE_REGEXP = '/^' . self::QNAME_WITH_DOMAIN_PART . '\-' . self::OID_PART . '$/i';
    const ID_REGEXP = '/^' . self::QNAME_WITH_DOMAIN_PART . '$/i';
    const URI_REGEXP = '/^' . self::DOMAIN_PART . self::QNAME_PART . '$/i';

    const EMPTY_NS_QNAME_PREFIX = self::QNAME_PREFIX_HERBO;
    const DEFAULT_DB_QNAME_PREFIX = self::QNAME_PREFIX_TUN;

    /** @var string default qname prefix without the trailing colon */
    private static $defaultQNamePrefix = 'luomus';

    private $autoGeneratedNamespaces = ['HR', 'HRA', 'MOS', 'MOS', 'HT', 'MY', 'GX', 'test', 'P', 'S'];
    private $id = null;
    private $ns = null;
    private $hasId = false;
    private $seqGen;
    private $defaultNS = [
        'MOS.organization' => 'MOS',
        'GX.dataset' => 'GX',
        'HRA.transaction' => 'HRA',
        'MY.collection' => 'HR',
        'MY.document' => 'HT',
        'MY.gathering' => 'MY',
        'MY.unit' => 'MY',
        'MY.identification' => 'MY',
        'MY.typeSpecimen' => 'MY',
        'MY.measurementClass' => 'MY',
        'MF.sample' => 'S',
        'MF.preparationClass' => 'S',
        'PUU.branch' => 'P',
        'PUU.event' => 'MY',
    ];

    /**
     * Stores the configs for id generator
     *
     * @param SequenceGeneratorInterface $seqGen
     */
    public function __construct(SequenceGeneratorInterface $seqGen = null)
    {
        $this->seqGen = $seqGen;
    }

    public static function getAllNS($asMap = false, $includeTunAsDefault = false) {
        $result = [];
        foreach(self::$conversions as $ns => $domain) {
            $ns = $includeTunAsDefault && $ns === 'tun:' ? '' : rtrim($ns, ':');
            if ($asMap) {
                $result[$domain] = $ns;
            } else {
                $result[] = $ns;
            }
        }
        return $result;
    }

    /**
     * Returns the full uri so that key 0 is the domain and key 1 is the rest
     *
     * @param $value
     * @return array
     */
    public static function getUriInParts($value)
    {
        $uri = self::getUri($value);
        $pos = strrpos($uri, '/');
        if ($pos === false) {
            return ['', $value];
        }
        return [
            substr($uri, 0, $pos + 1),
            substr($uri, $pos + 1),
        ];
    }

    public static function isUri($value)
    {
        return (strpos($value, 'http://') === 0);
    }

    public static function getUri($value, $useDefaultDomain = false)
    {
        if (is_array($value)) {
            return array_map(self::class . '::getUri', $value);
        }
        if (self::isUri($value)) {
            return $value;
        }
        $pos = strpos($value, ':');
        if ($pos !== false) {
            $domQName = substr($value, 0, $pos + 1);
            if (isset(self::$conversions[$domQName])) {
                return
                    self::$conversions[$domQName] .
                    substr($value, $pos + 1);

            }
        }
        if (strpos($value, '.') === false && $pos === false) {
            return self::getUri(self::EMPTY_NS_QNAME_PREFIX . self::getQNameWithoutPrefix($value));
        }
        if ($useDefaultDomain) {
            return self::getUri(self::getDefaultQNamePrefix() . ':' . self::getQNameWithoutPrefix($value));
        }
        return self::getUri(self::DEFAULT_DB_QNAME_PREFIX . self::getQNameWithoutPrefix($value));
    }

    public static function getQNameWithoutPrefix($value)
    {
        if (self::isUri($value)) {
            $value = str_replace(
                array_values(self::$conversions),
                array_keys(self::$conversions),
                $value
            );
        }
        $pos = strpos($value, ':');
        if ($pos !== false) {
            $prefix = substr($value, 0, $pos + 1);
            if (isset(self::$conversions[$prefix])) {
                return substr($value, $pos + 1);
            }
        }
        return $value;
    }

    protected static function getQNameDbStyle($value)
    {
        return self::getQName($value, true);
    }

    public static function getQName($value, $dbStyle = false)
    {
        if (is_array($value)) {
            if ($dbStyle) {
                return array_map(self::class . '::getQNameDbStyle', $value);
            }
            return array_map(self::class . '::getQName', $value);
        }
        if (strpos($value, 'http') === 0) {
            $value = str_replace(
                array_values(self::$conversions),
                array_keys(self::$conversions),
                $value
            );
        }
        $pos = strpos($value, ':');
        if ($pos === false) {
            $value = (strpos($value, '.') === false)?
                self::EMPTY_NS_QNAME_PREFIX . $value :
                self::DEFAULT_DB_QNAME_PREFIX . $value;
        }
        if ($dbStyle && strpos($value, self::DEFAULT_DB_QNAME_PREFIX) === 0) {
            return substr($value, strlen(self::DEFAULT_DB_QNAME_PREFIX));
        }
        return $value;
    }

    public static function hasQNamePrefix($prefix)
    {
        $prefix = rtrim($prefix,':') . ':';
        return isset(self::$conversions[$prefix]);
    }

    public static function isValidDefaultQNamePrefix($prefix) {
        return isset(self::$conversions[$prefix . ':']);
    }

    public static function setDefaultQNamePrefix($prefix)
    {
        $prefix = rtrim($prefix,':');
        if (!self::isValidDefaultQNamePrefix($prefix)) {
            throw new \Exception('Invalid QName prefix given "' . $prefix . '"');
        }
        self::$defaultQNamePrefix = $prefix;
    }

    public static function getDefaultQNamePrefix()
    {
        return self::$defaultQNamePrefix;
    }

    public static function getDotLess($value) {
        $pos = strpos($value, '.');
        if ($pos !== false) {
            return substr($value, $pos + 1);
        }
        $pos = strpos($value, ':');
        if ($pos !== false) {
            return substr($value, $pos + 1);
        }
        return $value;
    }

    public static function getRegExWithoutGroups($regEx) {
        $regEx = preg_replace('/\?P\<\w+\>/','', $regEx);
        $regEx = preg_replace('/\?\(\<\w+\>\).*?(\|\\\.)?/','.*?', $regEx);

        return $regEx;
    }

    public function clear()
    {
        $this->id = null;
        $this->ns = null;
        $this->hasId = false;
    }

    /**
     * Generates new id for the class to use and
     *
     * @param $namespace
     * @return IdService
     * @throws Exception\DomainException
     * @throws Exception\RuntimeException
     */
    public function generate($namespace)
    {
        if ($this->seqGen === null) {
            throw new Exception\RuntimeException("You have to set sequence generator before generating id");
        }
        $pos = strpos($namespace, ':');
        $ns = $namespace;
        if ($pos !== false) {
            $ns = substr($namespace, $pos + 1);
        } else {
            $namespace = self::$defaultQNamePrefix . ':' . $namespace;
        }

        if (!in_array($ns, $this->autoGeneratedNamespaces)) {
            throw new Exception\DomainException('IdGenerator must have valid namespace');
        }
        $this->id = $this->seqGen->getNextSeqVal($ns);
        $this->ns = $namespace;
        $this->hasId = true;

        return $this;
    }

    public function generateFromPieces($objId, $nsId, $type)
    {
        if (empty($type)) {
            throw new Exception\LogicException('Type should be set, but none was given');
        }
        if (StrUtils::isEmpty($objId) && StrUtils::isEmpty($nsId)) {
            $this->generate($this->getDefaultNamespace($type));
        } else {
            if (StrUtils::isEmpty($objId)) {
                throw new Exception\LogicException('Object ID should be set, but none was given');
            }
            $this->setId($objId);
            $ns = StrUtils::isEmpty($nsId) ? '' :
                (strpos($nsId, ':') === false ? self::$defaultQNamePrefix . ':' . $nsId : $nsId);
            $this->setNamespace($ns);
        }

        return $this;
    }

    private function getDefaultNamespace($type)
    {
        if (!isset($this->defaultNS[$type])) {
            throw new Exception\LogicException('No default namespace exists to type ' . $type);
        }
        return $this->defaultNS[$type];
    }

    /**
     * Returns the fully qualified domain name.
     * Example:
     *      http://id.luomus.fi/MA.157
     *
     * @return string
     */
    public function getFQDN()
    {
        return (string)self::getUri($this->getPQDN());
    }

    /**
     * Returns partially qualified domain name
     * Example:
     *      MA.157
     *
     * @return string
     */
    public function getPQDN()
    {
        $ns = $this->getNamespace();
        if (empty($ns) || $ns === self::EMPTY_NS_QNAME_PREFIX) {
            $part =  self::EMPTY_NS_QNAME_PREFIX . $this->getId();
        } else {
            $part = $ns . '.' . $this->getId();
        }
        return self::getQName($part);
    }

    /**
     * Return the used namespace
     *
     * @return string
     * @throws Exception\LogicException
     */
    public function getNamespace()
    {
        if (!$this->hasId) {
            throw new Exception\LogicException('You have to first generate the id');
        }
        return (string)$this->ns;
    }

    /**
     * Return the new id
     *
     * @return string
     * @throws Exception\LogicException
     */
    public function getId()
    {
        if (!$this->hasId) {
            throw new Exception\LogicException('You have to first generate the id');
        }
        return (string)$this->id;
    }

    /**
     * Returns the new id as integer
     * @return int
     * @throws Exception\LogicException
     */
    public function getNumericId()
    {
        if (!$this->hasId) {
            throw new Exception\LogicException('You have to first generate the id');
        }
        $id = preg_replace("/[^0-9]/", "", $this->id);

        return (int)$id;
    }

    /**
     * Sets the namespace
     *
     * @param $namespace
     * @return $this
     */
    public function setNamespace($namespace)
    {
        $this->ns = (string)trim($namespace);

        return $this;
    }

    /**
     * Sets the id
     *
     * @param $id
     * @return $this
     */
    public function setId($id)
    {
        $this->hasId = true;
        $this->id = (string)trim($id);

        return $this;
    }

}
