<?php

namespace Luomus\InputFilter;


use Common\Service\IdService;
use Triplestore\Model\Metadata;
use Triplestore\Model\Properties;
use Triplestore\Service\MetadataService;
use Zend\Code\Generator\ClassGenerator;
use Zend\Code\Generator\DocBlockGenerator;
use Zend\Code\Generator\FileGenerator;
use Zend\Code\Generator\MethodGenerator;

class Generator
{
    const STYLE_SHORT      = 'short';
    const STYLE_FULL       = 'full';
    const STYLE_FULL_CAMEL = 'full-camel';

    private $metadataService;
    private $propertyService;
    private $config;
    private $validatorSpecProvider;
    private $style = self::STYLE_SHORT;
    private $path;
    private $skippedProperties = [];

    public function __construct(MetadataService $metadataService, $config, ValidatorSpecProviderInterface $validatorSpecProvider)
    {
        $this->validatorSpecProvider = $validatorSpecProvider;
        $this->metadataService = $metadataService;
        $this->propertyService = $metadataService->getAllProperties();
        $this->config = $config;
        if (!isset($this->config['location'])) {
            throw new \Exception('Missing location for creating input filter');
        }
        if (isset($this->config['skipped_properties'])) {
            $this->skippedProperties = $this->config['skipped_properties'];
        }
        $this->path = rtrim($this->config['location'], '\\/') . '/';
        if (!is_writable($this->path)) {
            throw new \Exception('Location "' . $this->path . '" for creating input filter is not writable');
        }
    }

    /**
     * @return string
     */
    public function getStyle()
    {
        return $this->style;
    }

    /**
     * @param string $style
     */
    public function setStyle($style)
    {
        $this->style = $style;
    }

    public function generateValidators(array $classes = []) {
        $allClasses = $this->metadataService->getAllClasses();
        if (empty($classes)) {
            $classes = array_keys($allClasses);
        } else {
            $classes = array_filter($classes, function($v) {
                return IdService::getQName($v, true);
            });
        }
        $ns        = isset($this->config['namespace']) ? $this->config['namespace'] : null;
        $defExtend = isset($this->config['default_extend']) ? $this->config['default_extend'] : null;
        foreach($allClasses as $class => $metadata) {
            if (!in_array($class, $classes)) {
                continue;
            }
            $properties = $metadata->getProperties();
            if (count($properties) < 1) {
                continue;
            }
            /** @var Metadata $metadata */
            $extend = isset($this->config['extend'][$class]) ? $this->config['extend'][$class] : $defExtend;
            $this->generateValidator(
                $this->getNameInStyle($class, true),
                $ns,
                $extend,
                $metadata
            );
        }
    }

    protected function generateValidator($className, $ns, $extend, Metadata $metadata) {
        $validator = new ClassGenerator($className, $ns, null, $extend);
        $docblock  = DocBlockGenerator::fromArray([
            'shortDescription' => 'Validator class for ' . $className,
            'longDescription'  => "These will validate the user input based on definitions in triplestore",
        ]);
        $validator->setDocBlock($docblock);
        $validator->addUse('Zend\InputFilter\CollectionInputFilter');
        $validator->addUse('Luomus\InputFilter\BaseInputFilter');
        $validator->addMethodFromGenerator($this->initMethod($metadata, $ns));
        $fileGenerator = FileGenerator::fromArray(array(
            'classes'  => [$validator],
            'docblock' => DocBlockGenerator::fromArray(array(
                'shortDescription' => 'Automatically generated class',
                'longDescription'   => 'Do not edit this file this is automatically generated when classes are generated.'
            )),
        ));
        $file = $this->path . '/' . $className . '.php';
        file_put_contents($file, $fileGenerator->generate());
    }

    protected function initMethod(Metadata $metadata, $ns) {
        return new MethodGenerator(
            'init',
            [],
            MethodGenerator::FLAG_PUBLIC,
            $this->initBody($metadata, $ns)
        );
    }

    protected function initBody(Metadata $metadata, $ns) {
        $body = '';
        /** @var Properties $propertyHelper */
        $propertyHelper = $metadata->getAllProperties();
        $properties = $metadata->getProperties();

        foreach($properties as $property) {
            $type = $metadata->getType($property);
            if ($type === null || $type === 'rdfs:Resource' || in_array($property, $this->skippedProperties)) {
                continue;
            }
            if ($propertyHelper->isEmbeddable($property)) {
                $body .= "\n" . $this->validatorSpecProvider->getEmbeddedObjectSpec(
                        $this->getNameInStyle($type, true),
                        $this->getNameInStyle($property),
                        $metadata->hasMany($property),
                        $metadata->isRequired($property)
                    ) . "\n";
                continue;
            }
            $require = $metadata->isRequired($property) ? 'true' : 'false';
            $hasMane = $metadata->hasMany($property);
            $multiLanguage = $metadata->isMultiLanguage($property);
            $description = $metadata->getHelp($property);
            $description = addslashes($description);
            $body .= "\n\$this->add([\n";
            $body .= "  'name' => '" . $this->getNameInStyle($property) . "',\n";
            $body .= "  'required' => $require,\n";
            $body .= "  'inputType' => '" . ($hasMane ? "[$type]" : $type) . "',\n";
            $body .= "  'description' => '$description',\n";
            $body .= "  'validators' => [\n";
            $body .= $this->getValueValidators($property, $type, $hasMane, $multiLanguage) . ",\n";
            if ($require === 'false' && $type === 'xsd:boolean') {
                $body .= "  ['name' => 'notEmpty', 'options' => ['type' => 'null']]\n";
            }
            $body .= "  ]\n";
            $body .= "]);\n";
        }
        return $body;
    }

    protected function getNameInStyle($variable, $isClass = false) {
        if ($this->style === self::STYLE_FULL && !$isClass) {
            return $variable;
        } elseif (in_array($this->style, [self::STYLE_FULL, self::STYLE_FULL_CAMEL])) {
            return str_replace(' ', '', ucwords(str_replace([':','.'], ' ', $variable)));
        } elseif ($this->style == self::STYLE_SHORT) {
            $pos = strpos($variable, '.');
            if ($pos !== false) {
                $variable = substr($variable, $pos + 1);
            }
            $pos = strpos($variable, ':');
            if ($pos !== false) {
                $variable = substr($variable, $pos + 1);
            }
            if ($isClass) {
                $variable = ucfirst($variable);
            }
            return $variable;
        } else {
            throw  new \Exception(sprintf("Invalid style '%s' given! Expecting one of %s",
                $this->style,
                implode(', ', [self::STYLE_SHORT, self::STYLE_FULL, self::STYLE_FULL_CAMEL])
            ));
        }
    }

    protected function getValueValidators($property, $type, $hasMane, $multiLanguage) {
        $specProvider = $this->validatorSpecProvider;
        $spec = '';
        switch($type) {
            case 'xsd:decimal':
                $spec = $specProvider->getDecimalSpec();
                break;
            case 'xsd:string':
                $spec = $specProvider->getStringSpec();
                break;
            case 'xsd:dateTime':
                $spec = $specProvider->getDateTimeSpec();
                break;
            case 'xsd:date':
                $spec = $specProvider->getDateSpec();
                break;
            case 'xsd:boolean':
                $spec = $specProvider->getBooleanSpec();
                break;
            case 'xsd:gYear':
            case 'xsd:int':
            case 'xsd:integer':
                $spec = $specProvider->getIntegerSpec();
                break;
            case 'xsd:positiveInteger':
                $spec = $specProvider->getIntegerSpec(1);
                break;
            case 'xsd:nonNegativeInteger':
                $spec = $specProvider->getIntegerSpec(0);
                break;
            case 'xsd:anyURI':
                $spec = $specProvider->getAnyURISpec();
                break;
            case 'MZ.keyValue':
                $spec = $specProvider->getKeyValueSpec();
                break;
            case 'MZ.keyAny':
                $spec = $specProvider->getKeyAnySpec();
                break;
            case 'MZ.geometry':
                $spec = $specProvider->getGeometrySpec();
                break;
            default:
                if (strpos($type, 'xsd:') === 0) {
                    throw new \Exception("Invalid type '$type' specified to property '$property'");
                }
                if ($this->metadataService->altExists($type)) {
                    $spec = $specProvider->getAltSpec($type);
                } else {
                    if (!$this->metadataService->getAllProperties()->isEmbeddable($property)) {
                        $spec = $specProvider->getObjectExistsSpec($type);
                    }
                }
        }
        if (is_array($spec)) {
            if ($hasMane) {
                $spec = $specProvider->makeArray($spec);
            }
            if ($multiLanguage) {
                $spec = $specProvider->makeMultiLanguage($spec);
            }
            return var_export($spec, true);
        }
        return $spec;
    }

}