<?php

namespace Kotka\Controller;

use Common\Service\IdService;
use Kotka\Service\TransactionService;
use Kotka\Stdlib\Hydrator\OntologyHydrator;
use Kotka\Stdlib\Hydrator\Strategy\DateTimeStrategy;
use Kotka\Triple\HRATransaction;
use Kotka\Triple\MYDocument;
use Triplestore\Stdlib\Hydrator\DateTime;
use Zend\Http\Headers;
use Zend\Http\Response;
use Zend\Stdlib\Parameters;
use Zend\View\Model\ViewModel;

/**
 * This handles all the transactions
 *
 * @package Kotka\Controller
 */
class TransactionsController extends AbstractKotkaController
{
    const DEFAULT_PER_PAGE = 100;

    protected $filterKey = 'filter-transactions';

    /** @var \Kotka\Service\TransactionService */
    protected $documentService;
    /** @var \Kotka\Service\OrganizationService */
    protected $organizationService;

    protected $editTitle = 'Edit transaction';
    protected $addTitle = 'Create new transaction';

    /**
     * Handles the view with transaction (or loan) list
     * Template in view/transactions/index.phtml
     *
     * @return ViewModel
     */
    public function indexAction()
    {
        return new ViewModel(array('transactions' => $this->getDocumentService()->getAll()));
    }

    /**
     * Adds the new transaction
     * @return ViewModel
     */
    public function addAction()
    {
        $view = parent::addAction();

        return $view;
    }

    public function exportExcelAction()
    {
        $file = './data/generated_excels/transactions.xlsx';
        $transactions = $this->getDocumentService()->getAll();
        /** @var \Kotka\Service\ExcelService $excelService */
        $excelService = $this->getServiceLocator()->get('Kotka\Service\ExcelService');
        $excelService->getEventManager()->attach('pre-add', function ($e) {
            $params = $e->getParams();
            $data = &$params['data'];
            if (isset($data['HRBMissing'])) {
                $data['HRBMissing'] = count($data['HRBMissing']);
            }
            if (isset($data['HRBDamaged'])) {
                $data['HRBDamaged'] = count($data['HRBDamaged']);
            }
            if (isset($data['HRBReturned'])) {
                $data['HRBReturned'] = count($data['HRBReturned']);
            }
            if (isset($data['HRBAway'])) {
                $data['HRBAway'] = count($data['HRBAway']);
            }
            unset($data['MZPublicityRestrictions']);
        });
        $form = $this->getEditForm();
        $form->addMetaFields();
        $excelService->setForm($form);
        $excelService->generateExcelFromArray($transactions, $file, true);
        return $excelService->getResponse();
    }

    public function exportSpecimenExcelAction()
    {
        /** @var \Kotka\Service\SpecimenService $specimenService */
        /** @var \Kotka\Service\ExcelService $excelService */
        $uri = $this->getRequest()->getQuery('uri');
        $transaction = $this->getDocumentService()->getById($uri);
        $loanItems = $transaction->getHRBAway();
        $itemCnt = count($loanItems);
        if (is_null($loanItems) || count($loanItems) === 0) {
            return array('title' => 'No specimen data!', 'msg' => "Couldn't find any specimen that has away status.");
        }
        if ($itemCnt > SearchController::MAX_EXPORT) {
            return array('title' => 'Too many rows to export!', 'msg' => "You cannot export to Excel if you have more than " . SearchController::MAX_EXPORT . " rows.");
        }
        $specimenService = $this->serviceLocator->get('Kotka\Service\SpecimenService');
        $specimens = $specimenService->getById($loanItems);
        if (count($specimens) !== $itemCnt) {
            $specimens = array_merge($specimens, $specimenService->getById(array_map(function($item) {
                return 'luomus:' . $item;
            }, $loanItems)));
        }
        if (is_null($specimens) || count($specimens) === 0) {
            return array('title' => 'No specimen data!', 'msg' => "Couldn't find any specimen information in this loan data.");
        }
        $hydrator = new OntologyHydrator();
        $dateStrategy = new DateTimeStrategy();
        $dateStrategy->setFormat(DateTime::FORMAT);
        $hydrator->addStrategy('MZDateEdited', $dateStrategy);
        $specimenList = array();
        foreach ($specimens as $specimen) {
            $specimenList[] = $hydrator->extract($specimen);
        }
        $excelService = $this->serviceLocator->get('Kotka\Service\ExcelService');
        $filename = $this->getFileName($uri);
        $fullFilename = $this->getPath() . $filename;
        $excelService->setForm($this->getSpecimenForm());
        $excelService->generateExcelFromArray($specimenList, $fullFilename, true);
        if (!file_exists($fullFilename)) {
            return array('title' => 'File creation failed!', 'msg' => "Kotka was unable to create excel file.");
        }
        $file = file_get_contents($fullFilename);
        $headers = new Headers();
        $response = new Response();
        $headers->clearHeaders()
            ->addHeaderLine('Content-type', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet')
            ->addHeaderLine('Content-Disposition', 'attachment; filename="' . $filename . '"')
            ->addHeaderLine("Content-Length", strlen($file));
        $response->setHeaders($headers);
        $response->setContent($file);

        return $response;
    }

    private function getFileName($uri)
    {
        $file = 'Loan-' . IdService::getQName($uri);
        $file = str_replace(['.', ':'], '_', $file);
        $file .= '.xlsx';
        return $file;
    }

    private function getPath()
    {
        $config = $this->serviceLocator->get('Config');
        return $config['excel']['directory'] . DIRECTORY_SEPARATOR;
    }

    /**
     * Edits the existing transaction
     * @return ViewModel
     */
    public function editAction()
    {
        return parent::editAction();
    }

    protected function getAndStoreQuery()
    {
        $params = $this->params()->fromQuery();
        if ($params == null) {
            $sParams = $this->Store()->fetch($this->filterKey);
            if ($sParams !== null) {
                $params = new Parameters($sParams);
                $this->getRequest()->setQuery($params);
            }
        } else {
            $this->Store($this->filterKey, $params);
        }
        return $params;
    }

    protected function postValidate($document, $postData, $type)
    {
        if (!$document instanceof HRATransaction) {
            return;
        }
        if (isset($postData['HRBAway'])) {
            $document->setHRBAway(explode(', ', $postData['HRBAway']));
        }
        if (isset($postData['HRBDamaged'])) {
            $document->setHRBDamaged(explode(', ', $postData['HRBDamaged']));
        }
        if (isset($postData['HRBMissing'])) {
            $document->setHRBMissing(explode(', ', $postData['HRBMissing']));
        }
        if (isset($postData['HRBReturned'])) {
            $document->setHRBReturned(explode(', ', $postData['HRBReturned']));
        }
    }

    protected function preValidate($document, &$data, $type)
    {
        if (!isset($data['HRAPermit'])) {
            $data['HRAPermit'] = [];
        }
        parent::preValidate($document, $data, $type);
    }

    protected function postValidateError($document, &$data, $type)
    {
        $transaction = $this->getDocumentService();
        if (isset($data['HRAPermit']) && is_array($data['HRAPermit'])) {
            foreach ($data['HRAPermit'] as &$permit) {
                if (isset($permit['HRAPermitFile']) && is_array($permit['HRAPermitFile']) && !empty($permit['HRAPermitFile']['name'])) {
                    $id = uniqid();
                    $permit['HRAPermitFile']['name'] = TransactionService::TMP_FILE_PREFIX . $id . TransactionService::TMP_FILE_INFO_SEP . $permit['HRAPermitFile']['name'];
                    $permit['HRAPermitFile'] = $transaction->storeTmpFile($permit['HRAPermitFile']['name'], $permit['HRAPermitFile']['tmp_name'], 'tmp', $id);
                }
            }
        }
    }

    /**
     * @return \Kotka\Service\OrganizationService
     */
    protected function getOrganizationService()
    {
        if ($this->organizationService === null) {
            $this->organizationService = $this->serviceLocator->get('Kotka\Service\OrganizationService');
        }
        return $this->organizationService;
    }

    /**
     * Returns the transaction service
     *
     * @return \Kotka\Service\TransactionService
     */
    protected function getDocumentService()
    {
        if ($this->documentService === null) {
            $this->documentService = $this->serviceLocator->get('Kotka\Service\TransactionService');
        }

        return $this->documentService;
    }

    /**
     * Return form for editing transaction
     *
     * @return \Kotka\Form\HRATransaction
     */
    protected function getEditForm()
    {
        /** @var \Kotka\Service\FormGeneratorService $formGenerator */
        $formGenerator = $this->serviceLocator->get('Kotka\Service\FormGeneratorService');
        $form = $formGenerator->getForm(new HRATransaction());

        return $form;
    }

    /**
     * Return form for adding transaction
     * @return \Kotka\Form\HRATransaction
     */
    protected function getAddForm()
    {
        $form = $this->getEditForm();
        $form->setObject(new HRATransaction());

        return $form;
    }

    protected function getEditTitle()
    {
        return $this->editTitle . ' ' . $this->document->getUri();
    }

    private function getSpecimenForm()
    {
        $formGenerator = $this->serviceLocator->get('Kotka\Service\FormGeneratorService');
        $form = $formGenerator->getForm(new MYDocument());

        return $form;
    }

}
