<?php

namespace Kotka\DataGrid;

use Kotka\Service\FormElementService;
use Kotka\Service\OrganizationService;
use Zend\Form\Form;
use ZfcDatagrid\Column;
use ZfcDatagrid\Column\Style;
use ZfcDatagrid\Column\Type;
use ZfcDatagrid\Filter;


class TransactionGrid extends AbstractGrid
{

    private $organizationService;
    private $elementService;

    public function __construct(OrganizationService $organizationService, FormElementService $elementService)
    {
        $this->organizationService = $organizationService;
        $this->elementService = $elementService;
    }

    public function prepareGrid($data, $instructions = null, $perPage = null)
    {
        $perPage = $perPage !== null ? $perPage : $this->perPage;
        $this->grid->setTitle('Transactions');
        $this->grid->setId('loans');
        $this->grid->setDataSource($data);
        $this->grid->setDefaultItemsPerPage($perPage);
        $this->grid->setUserFilterDisabled(true);
        foreach ($instructions as $instruction) {
            if (!isset($instruction['source'])) {
                throw new \Exception('You are missing instructions for dataGrid');
            }
            $label = $instruction['source'];
            if (isset($instruction['label'])) {
                $label = $instruction['label'];
            }
            $column = new Column\Select($instruction['source']);
            $column->setLabel($label);
            $column->setFilterDefaultOperation(Filter::EQUAL);
            if (isset($instruction['class'])) {
                foreach ($instruction['class'] as $class => $info) {
                    $style = new Style\RowClass($class);
                    $column->addStyle($style);
                    if (!is_array($info)) {
                        continue;
                    }
                    if (isset($info['value'])) {
                        $filter = isset($info['filter']) ? $info['filter'] : Filter::EQUAL;
                        $style->addByValue($column, $info['value'], $filter);
                    }
                }
            }
            if (isset($instruction['width'])) {
                $column->setWidth($instruction['width']);
            } else {
                $column->setWidth(10);
            }
            if (isset($instruction['hidden'])) {
                $column->setHidden($instruction['hidden']);
            }
            if (isset($instruction['convert'])) {
                $column->setReplaceValues($instruction['convert']);
            }
            if (isset($instruction['type'])) {
                if (!is_array($instruction['type']) || !isset($instruction['type']['class'])) {
                    throw new \Exception("When using type you should at least have class key in the type");
                }
                $type = new $instruction['type']['class']();
                if (isset($instruction['type']['params'])) {
                    $params = $instruction['type']['params'];
                    foreach ($params as $method => $value) {
                        $type->$method($value);
                    }
                }
                $column->setType($type);
            }
            $this->grid->addColumn($column);
        }
    }

    public function setFilters($params, Form $form)
    {
        if ($params == null) {
            return;
        }
        $filters = array();
        foreach ($params as $field => $value) {
            if ($value === '' || $value === null) {
                continue;
            }
            if ($form->has($field)) {
                $col = $this->grid->getColumnByUniqueId($field);
                if ($col === null) {
                    continue;
                }
                $filter = new Filter();
                $filter->setFromColumn($col, $value);
                $filters[] = $filter;
            }
        }
        $this->grid->getRenderer()->setFilters($filters);
    }

    public function extractData($transactions, $editUrl)
    {
        $result = array();
        $os = $this->organizationService;
        foreach ($transactions as $transaction) {
            $data = array();
            /** @var \Kotka\Triple\HRATransaction $transaction */
            $data['URI'] = $transaction->getUri();
            $data['Owner'] = $transaction->getMZOwner();
            $data['Type'] = $transaction->getHRATransactionType();
            $organization = $transaction->getHRACorrespondentOrganization();
            if (!empty($organization)) {
                $organization = $os->getOrganizationName($os->getById($organization));
            }
            $data['Organization'] = $organization;
            $data['SenderId'] = $transaction->getHRAExternalTransactionID();
            $data['Person'] = $transaction->getHRACorrespondentPerson();
            $data['Received'] = $transaction->getHRATransactionRequestReceived();
            $returned = count($transaction->getHRBReturned()) + $transaction->getHRBReturnedOther();
            $away = count($transaction->getHRBAway()) + $transaction->getHRBAwayOther();
            $missing = count($transaction->getHRBMissing()) + $transaction->getHRBMissingOther();
            $damaged = count($transaction->getHRBDamaged()) + $transaction->getHRBDamagedOther();
            $total = $away + $missing + $damaged + $returned;
            $data['Total'] = $total;
            $return = array();
            if (($returned + $damaged) > 0) {
                $return[] = $returned + $damaged;
            }
            if ($missing > 0) {
                $return[] = '(' . $missing . ')';
            }
            $data['Returned'] = implode(' ', $return);
            $data['Balance'] = $away * -1;
            $dueDate = $transaction->getHRADueDate();
            $data['Due'] = '';
            $data['DueDate'] = '';
            if ($dueDate instanceof \DateTime && $away > 0) {
                $now = new \DateTime();
                $interval = $now->diff($dueDate);
                $data['Due'] = intval($interval->format('%r%a'));
                $data['DueDate'] = $interval->format('%y y %m m %d d');
            }
            $data['Status'] = $transaction->getHRATransactionStatus();
            $result[] = $data;
        }
        $options = $this->elementService->getSelect('HRA.transactionType');
        $statuses = $this->elementService->getSelect('HRA.transactionStatus');
        $instructions = array(
            array(
                'source' => 'URI',
                'type' => array(
                    'class' => '\\Kotka\\DataGrid\\Type\\URI',
                    'params' => array(
                        'setEditUrl' => $editUrl,
                        'setUseFullUrl' => false,
                    )
                ),
            ),
            array(
                'source' => 'Owner',
                'hidden' => true,
            ),
            array(
                'source' => 'Type',
                'convert' => $options,
                'class' => array(
                    'direction-incoming' => array(
                        'value' => 'Incoming',
                        'filter' => Filter::LIKE,
                    ),
                    'direction-outgoing' => array(
                        'value' => 'Outgoing',
                        'filter' => Filter::LIKE,
                    ),
                    'type-loan' => array(
                        'value' => 'Loan',
                        'filter' => Filter::LIKE
                    ),
                    'type-gift' => array(
                        'value' => 'Gift',
                        'filter' => Filter::LIKE
                    ),
                    'type-exchange' => array(
                        'value' => 'Exchange',
                        'filter' => Filter::LIKE
                    )
                )
            ),
            array(
                'source' => 'Status',
                'convert' => $statuses,
            ),
            array(
                'source' => 'Organization',
                'width' => 100
            ),
            array(
                'source' => 'Person',
                'width' => 100
            ),
            array(
                'source' => 'SenderId',
                'label' => 'Sender loan id',
            ),
            array(
                'source' => 'Received',
                'type' => array(
                    'class' => '\\ZfcDatagrid\\Column\\Type\\DateTime'
                )
            ),
            array(
                'source' => 'Total',
                'type' => array(
                    'class' => '\\ZfcDatagrid\\Column\\Type\\Number'
                ),
                'class' => array(
                    'total-empty' => array(
                        'value' => '',
                    )
                ),
            ),
            array(
                'source' => 'Returned',
                'label' => 'Returned (missing)',
                'width' => 1,
                'class' => array(
                    'returned-empty' => array(
                        'value' => ''
                    )
                )
            ),
            array(
                'source' => 'Balance',
                'type' => array(
                    'class' => '\\ZfcDatagrid\\Column\\Type\\Number'
                ),
                'class' => array(
                    'balance-0' => array(
                        'value' => 0,
                        'filter' => Filter::EQUAL,
                    ),
                    'balance-negative' => array(
                        'value' => 0,
                        'filter' => Filter::LESS,
                    )
                )
            ),
            array(
                'source' => 'Due',
                'label' => 'Due days',
                'type' => array(
                    'class' => '\\ZfcDatagrid\\Column\\Type\\Number'
                ),
                'class' => array(
                    'due-negative' => array(
                        'value' => 0,
                        'filter' => Filter::LESS
                    ),
                    'due-0' => array(
                        'value' => 0,
                    ),
                    'due-positive' => array(
                        'value' => 0,
                        'filter' => Filter::GREATER
                    ),
                    'due-expiring' => array(
                        'value' => 31,
                        'filter' => Filter::LESS
                    )
                )
            ),
            array(
                'source' => 'DueDate',
                'label' => 'Due d/m/y',
            )
        );

        return array($result, $instructions);
    }

}